########################################################################
# R code: Section 10.2.1
# File: LS-PI-forecast.r 
# Coded by: Marcella Niglio
#
# Plug-in and Least Squares forecasts from SETARMA models
# 
# Reference:
# Niglio, M. (2007).
#   Multi-step forecasts from threshold ARMA models using asymmetric 
#   loss functions.
#   Statistical Methods & Applications, 16(3), 395-410.
#   DOI: 10.1007/s10260-007-0044-x.
########################################################################
#
prev.SETARMA <- function(dati,datiF,errori1,errori2,var1,var2,h,AR.first,MA.first,AR.second,MA.second,AR.third,MA.third,d,v.s1,v.s2){
#
# Generates Least Squares (LS) and Plug-in (PI) forecasts from a 3-regime SETARMA model
#
# N         = series length
# dati      = vector of data
# datiF     = ex-post data
# errori1   = vector of first regime errors
# errori2   = vector of second regime errors
# var1/2    = errors variance of the first/second regime
# h         = lead time
# AR.first  = vector of the AR coefficients of the first regime 
#             (intercept included as first element of the vector);
# MA.first  = vector of he the MA coefficients of the first regime;
# AR.second = vector of the AR coefficients of the second regime 
#             (intercept included as first element of the vector);
# MA.second = vector of he the MA coefficients of the second regime;
# AR.third  = vector of the AR coefficients of the third regime 
#             (intercept included as first element of the vector);
# MA.third  = vector othe the MA coefficients of the third regime;
# d         = threshold delay;
# v.s1      = threshold value 1
# v.s2      = threshold value 2
#
# INITIALIZATION
p  <- length(AR.first)-1 
q0 <- length(MA.first)
N  <- length(dati)

primo.for   <- vector("numeric", h)
secondo.for <- vector("numeric", h)
m.prev.tot  <- matrix(0,ncol=h,nrow=8,dimnames=list(c("oss","prev.LS","err.prev.LS","prob.trans","var.err.LS","prev.PI","err.prev.PI","var.err.PI"),NULL))
# m.prev.tot is a matrix with 8 rows:

# [1,]: ex-post data; 
# [2,]: forecasts LS; 
# [3,]: forecasts errors LS;
# [4,]: value of the indicator function/transition probability
# [5,]: variance prediction errors PI.
# [6,]: forecasts PI; 
# [7,]: forecast errors PI; 
# [8,]: variance prediction errors PI.
#
m.prev.tot[1,] <- datiF
pr.emp         <- 1 
ind            <- as.matrix(seq(from=1,to=(q0+1),by=1))
matr.trian     <- apply(ind,1,FUN=function(ii,qq0=q0+1){c(rep(1,ii-1),rep(0,qq0-ii+1))})
vett.temp.pos  <- c(0, errori1[length(errori1):(length(errori1)-q0+1)])
vett.temp.neg  <- c(0, errori2[length(errori2):(length(errori2)-q0+1)])

matr.prev.pos <- matrix(0,ncol=q0+1,nrow=h)
matr.prev.neg <- matrix(0,ncol=q0+1,nrow=h)
i <- 1
for(i in 1:(h)){
	if (i <=(q0)){
	matr.prev.pos[i,] <- matr.trian[i,]*vett.temp.pos[1:(q0+1)]
	matr.prev.neg[i,] <- matr.trian[i,]*vett.temp.neg[1:(q0+1)]
	vett.temp.pos     <- c(0,vett.temp.pos)
        vett.temp.neg     <- c(0,vett.temp.neg)}
	else{
	matr.prev.pos[i,] <- 0
        matr.prev.neg[i,] <- 0}
		}
        vect.coeff.1 <- c(AR.first,1,MA.first)
        vect.coeff.2 <- c(AR.second,1,MA.second)
        vect.coeff.3 <- c(AR.third,1,MA.third)
        val.oss      <- dati[N:(N-20)]
        psi.list     <- pesi(h,d,AR.first,MA.first,AR.second,MA.second,AR.third,MA.third)
i <- 1
for(i in 1:h){
	vett.prev.primo   <- c(1,val.oss[1:p],matr.prev.pos[i,])
	vett.prev.secondo <- c(1,val.oss[1:p],matr.prev.neg[i,])
	primo.for[i]      <- vect.coeff.1%*%vett.prev.primo
	secondo.for[i]    <- vect.coeff.2%*%vett.prev.secondo
	if((i<=d) && (val.oss[d]>=v.s)){	# CASE h<=d
		forec <- primo.for[i]
		m.prev.tot[4,i]     <- 1
		m.prev.tot[5,i]     <- var1*sum(psi.list$psi1[1:i]^2)
		m.prev.tot[2,i]     <- forec
		m.prev.tot[3,i]     <- m.prev.tot[1,i]-forec  # FORECAST ERROR
		m.prev.tot[(6:8),i] <- c(forec, m.prev.tot[3,i],m.prev.tot[5,i])
	}
	if((i<=d) && (val.oss[d]<v.s)){
		forec <- secondo.for[i]
		m.prev.tot[5,i]     <- var2*sum(psi.list$psi2[1:i]^2)
		val.oss             <- c(forec, val.oss)
		m.prev.tot[2,i]     <- forec
		m.prev.tot[3,i]     <- m.prev.tot[1,i]-forec   # FORECAST ERROR
		m.prev.tot[(6:8),i] <- c(forec, m.prev.tot[3,i],m.prev.tot[5,i])
	}
	
     if(i==(d+1)){ 		# CASE h=d+1
     vett.prev.primo   <- c(1, val.oss[1:p], matr.prev.pos[i,])
     vett.prev.secondo <- c(1, val.oss[1:p], matr.prev.neg[i,])
     primo.for[i]      <- vect.coeff.1%*%vett.prev.primo
     secondo.for[i]    <- vect.coeff.2%*%vett.prev.secondo
     sig.e.1           <- var1*sum(psi.list$psi1[1:i]^2)
     sig.e.2           <- var2*sum(psi.list$psi2[1:i]^2)
     sig.x.1           <- var1*sum(psi.list$psi1[(i+1):length(psi.list$psi1)]^2)
     sig.x.2           <- var2*sum(psi.list$psi2[(i+1):length(psi.list$psi2)]^2)
     sig.x.12          <- sqrt(var1*var2)*sum(psi.list$psi1[(i+1):length(psi.list$psi1)]*psi.list$psi2[(i+1):length(psi.list$psi2)])
     m.prev.tot[4,i]   <- lambda(c(dati, m.prev.tot[2,(1:(i-1))]),v.s)
     m.prev.tot[2,i]   <- m.prev.tot[4,i]*primo.for[i]+(1-m.prev.tot[4,i])*secondo.for[i]
     m.prev.tot[3,i]   <- m.prev.tot[1,i]-m.prev.tot[2,i]	 
     m.prev.tot[5,i]   <- sig.e.2+m.prev.tot[4,i]*(sig.e.1-sig.e.2)+(m.prev.tot[4,i]-m.prev.tot[4,i]^2)*(sig.x.1+sig.x.2-2*sig.x.12)
     forec             <- m.prev.tot[2,i]
     if (val.oss[d]>=v.s)
	forec.PI <- primo.for[i]
     else {forec.PI<-secondo.for[i]}
	val.oss.PI      <- c(forec.PI, val.oss)
	m.prev.tot[6,i] <- forec.PI
	m.prev.tot[7,i] <- m.prev.tot[1,i]-forec.PI
	m.prev.tot[8,i] <- sig.e.2+m.prev.tot[4,i]*(sig.e.1-sig.e.2)+(m.prev.tot[4,i]-m.prev.tot[4,i]^2)*(sig.x.1+sig.x.2-2*sig.x.12)
      }
      
    if(i>(d+1)){ 		# CASE h>d+1
    vett.prev.primo   <- c(1, val.oss[1:p], matr.prev.pos[i,])
    vett.prev.secondo <- c(1, val.oss[1:p], matr.prev.neg[i,])
    primo.for[i]      <- vect.coeff.1%*%vett.prev.primo
    secondo.for[i]    <- vect.coeff.2%*%vett.prev.secondo
    sig.e.1           <- var1*sum(psi.list$psi1[1:i]^2)
    sig.e.2           <- var2*sum(psi.list$psi2[1:i]^2)
    sig.x.1           <- var1*sum(psi.list$psi1[(i+1):length(psi.list$psi1)]^2)
    sig.x.2           <- var2*sum(psi.list$psi2[(i+1):length(psi.list$psi2)]^2)
    sig.x.12          <- sqrt(var1*var2)*sum(psi.list$psi1[(i+1):length(psi.list$psi1)]*psi.list$psi2[(i+1):length(psi.list$psi2)])
    m.prev.tot[4,i]   <- lambda(c(dati, m.prev.tot[2, (1:(i-1))]), v.s)
    m.prev.tot[2,i]   <- m.prev.tot[4,i]*primo.for[i]+(1-m.prev.tot[4,i])*secondo.for[i]
    m.prev.tot[3,i]   <- m.prev.tot[1,i]-m.prev.tot[2,i]	  
    m.prev.tot[5,i]   <- sig.e.2+m.prev.tot[4,i]*(sig.e.1-sig.e.2)+(m.prev.tot[4,i]-m.prev.tot[4,i]^2)*(sig.x.1+sig.x.2-2*sig.x.12)
    forec             <- m.prev.tot[2,i]
    if (val.oss.PI[d]>=v.s)
	forec.PI<-primo.for[i]
    else {forec.PI      <- secondo.for[i]}
	val.oss.PI      <- c(forec.PI, val.oss.PI)
	m.prev.tot[6,i] <- forec.PI
	m.prev.tot[7,i] <- m.prev.tot[1,i]-forec.PI
	m.prev.tot[8,i] <- sig.e.2+m.prev.tot[4,i]*(sig.e.1-sig.e.2)+(m.prev.tot[4,i]-m.prev.tot[4,i]^2)*(sig.x.1+sig.x.2-2*sig.x.12)
     }
    val.oss <- c(forec,val.oss)
  }

return(list(MatricePrevisioni=m.prev.tot,prev.primo.regime=primo.for,prev.secondo.regime=secondo.for))
# the list returned has the following objects:
#   MatricePrevisioni:   matrix with forecasts (see the content of m.prev.tot)
#   prev.primo.regime:   first regime forecasts
#   prev.secondo.regime: second regime forecasts
}
#
#
combina <- function(l.for,h,d,v.s)
{
# This procedure allows to obtain the forecasts combination:
	# l.for: list obtained from prev.SETARMA
	# h:     lead time
	# d:     threshold delay
	# v.s:   threshold value
	#
m.f   <- l.for$MatricePrevisioni
for.c <- vector("numeric", length=h)
for.c[1:d] <- m.f[2, (1:d)]
	agm.set <- c(m.f[2,(1:d)],m.f[6,(d+1):h])
	ii <- 1
	for(ii in (d+1):h){
		if (agm.set[ii-d]>=v.s)
		   for.c[ii] <- m.f[6,ii]
		else (for.c[ii]<-m.f[2, ii])
	agm.set[ii] <- for.c[ii]}
        err.c       <- m.f[1,]-for.c

return(list(forComb=for.c,erroriComb=err.c))
# OUTPUT: a list with the following elements:
#   forComb    = forecasts combination
#   erroriComb = errors forecasts combination
}

#
pesi <- function(h,d, AR.first, MA.first, AR.second, MA.second){
#
# h:         lead time;
# d:         threshold delay;
# AR.first:  vector of the AR coefficients of the first regime 
#            (intercept included as first element of the vector);
# MA.first:  vector of the the MA coefficients of the first regime;
# AR.second: vector of the AR coefficients of the second regime 
#            (intercept included as first element of the vector);
# MA.second: vector of the the MA coefficients of the second regime;
#
# This procedure allows to obtain the weights to estimate the errors variance
#
#
psi.1   <- vector("numeric",h)
psi.2   <- vector("numeric",h)
dimens  <- h-(length(AR.first)-1)

if(dimens > 1) rip <- rep(0,dimens) else rip <- 0
matr.coeff.1 <- matrix(c(AR.first[2:length(AR.first)],rip,MA.first,rip[1:(dimens-1)]),ncol=(length(AR.first)-1+length(rip)),nrow=2,byrow=T)
matr.coeff.2 <- matrix(c(AR.second[2:length(AR.second)],rip,MA.second,rip[1:(dimens-1)]),ncol=(length(AR.second)-1+length(rip)),nrow=2,byrow=T)
psi.temp.1   <- c(1)
psi.temp.2   <- c(1)
ii <- 1

for (ii in 2:h){
  ll         <- length(psi.temp.1)
  t1         <- sum(psi.temp.1[1:ll]*matr.coeff.1[1,1:ll])-matr.coeff.1[2,(ii-1)]
  t2         <- sum(psi.temp.2[1:ll]*matr.coeff.2[1,1:ll])-matr.coeff.2[2,(ii-1)]
  psi.temp.1 <- c(t1, psi.temp.1)
  psi.temp.2 <- c(t2, psi.temp.2)}
  psi.1      <- psi.temp.1[length(psi.temp.1):1]
  psi.2      <- psi.temp.2[length(psi.temp.2):1]
  return(list(psi1=psi.1,psi2=psi.2))
# psi1: first regime weigths
# psi2: second regime weigths
}
#
#
####################
# Empirical computation of the transition probability (when h>d)
####################
#
lambda <- function(serie,r){
#
# INPUT:
#  serie: vactor of data (obtained as c(dati,datiF));
#  r: threshold value;
#
	nreg <- 0
	N    <- length(serie)
	s    <- 1
	while(s <= N) {
		if(serie[s] >= r)
			nreg <- nreg + 1
		           s <- s+1
	}
	return(nreg/N)
}
